<?php

namespace Database\Seeders;

use App\Models\QuestionnaireConfig;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class QuestionnaireConfigSeeder extends Seeder
{
    public function run(): void
    {
        // Disable foreign key checks for seeding
        DB::statement('SET FOREIGN_KEY_CHECKS=0');
        
        // Clear existing data
        QuestionnaireConfig::truncate();
        
        // Create Manulife questionnaire config
        $manulifeConfig = QuestionnaireConfig::create([
            'name' => 'Manulife Medical Questionnaire (Ages 60+)',
            'slug' => 'manulife-medical-questionnaire-60-plus',
            'insurance_company' => 'Manulife',
            'min_age' => 60,
            'max_age' => 120,
            'is_active' => true,
            'step1_config' => $this->getManulifeStep1Config(),
            'step2_config' => $this->getManulifeStep2Config(),
            'step3_config' => $this->getManulifeStep3Config(),
            'step4_config' => $this->getManulifeStep4Config(),
            'rate_a_multiplier' => 1.00,
            'rate_b_multiplier' => 1.50,
            'rate_c_multiplier' => 2.50,
            'smoking_surcharge_percent' => 10.00,
            'description' => 'Standard Manulife medical questionnaire for applicants aged 60 and above. Based on Manulife Single-Trip Emergency Medical insurance requirements.',
        ]);
        
        // Create JF questionnaire config
        $jfConfig = QuestionnaireConfig::create([
            'name' => 'JF TOP Simplified Questionnaire',
            'slug' => 'jf-top-simplified-questionnaire',
            'insurance_company' => 'JF',
            'min_age' => 60,
            'max_age' => 120,
            'is_active' => true,
            'step1_config' => $this->getJfStep1Config(),
            'step2_config' => null,
            'step3_config' => null,
            'step4_config' => $this->getJfStep4Config(),
            'rate_a_multiplier' => 1.00,
            'rate_b_multiplier' => 1.25,
            'rate_c_multiplier' => 1.75,
            'smoking_surcharge_percent' => 15.00,
            'description' => 'Simplified health questionnaire for JF TOP Travel Insurance. No complex medical underwriting required.',
        ]);
        
        // Create Allianz questionnaire config (if needed)
        $allianzConfig = QuestionnaireConfig::create([
            'name' => 'Allianz Travel Medical Questionnaire',
            'slug' => 'allianz-travel-medical-questionnaire',
            'insurance_company' => 'Allianz',
            'min_age' => 65,
            'max_age' => 85,
            'is_active' => true,
            'step1_config' => $this->getAllianzStep1Config(),
            'step2_config' => $this->getAllianzStep2Config(),
            'step3_config' => null,
            'step4_config' => $this->getAllianzStep4Config(),
            'rate_a_multiplier' => 1.00,
            'rate_b_multiplier' => 1.30,
            'rate_c_multiplier' => 2.00,
            'smoking_surcharge_percent' => 12.00,
            'description' => 'Allianz travel insurance medical questionnaire for seniors.',
        ]);
        
        // Re-enable foreign key checks
        DB::statement('SET FOREIGN_KEY_CHECKS=1');
        
        $this->command->info('Seeded 3 questionnaire configurations: Manulife, JF, Allianz');
    }
    
    private function getManulifeStep1Config(): array
    {
        return [
            'title' => 'Eligibility Check - Step 1',
            'description' => 'Determine if you are eligible for Manulife insurance',
            'instructions' => 'Answer all questions honestly. Any "Yes" answer in this step means you cannot purchase this insurance.',
            'questions' => [
                [
                    'id' => 'q1_1',
                    'code' => 'heart_surgery_10y',
                    'text' => 'Have you had a heart bypass, coronary angioplasty, or heart valve surgery more than 10 years ago?',
                    'type' => 'yes_no',
                    'required' => true,
                    'help_text' => 'If unsure, consult your doctor before answering.',
                ],
                [
                    'id' => 'q1_2',
                    'code' => 'multiple_conditions_3y',
                    'text' => 'In the last 3 years, have you been diagnosed with, prescribed medication for, or treated for ANY TWO of: Heart condition, Lung condition, Stroke/CVA or mini-stroke/TIA, Diabetes, Narrowed or blocked artery in the legs?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q1_3',
                    'code' => 'heart_failure_2y',
                    'text' => 'In the last 2 years, have you been diagnosed with, prescribed medication for, or treated for heart failure/congestive heart failure?',
                    'type' => 'yes_no',
                    'required' => true,
                    'help_text' => 'Includes use of Lasix, furosemide, or other water pill for ankle/leg swelling or fluid in the lungs',
                ],
                [
                    'id' => 'q1_4',
                    'code' => 'recent_health_issues_12m',
                    'text' => 'In the last 12 months, have you had any of: New heart condition or change in medication/hospitalization for existing heart condition; Shortness of breath or chest pain requiring medical treatment; Lung condition with hospitalization or prednisone prescription; Cancer treatment (excluding certain skin cancers)?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q1_5',
                    'code' => 'multiple_medications_4m',
                    'text' => 'In the last 4 months, have you taken 6 or more prescription medications?',
                    'type' => 'yes_no',
                    'required' => true,
                    'help_text' => 'Excludes hormone replacement therapy, osteoporosis medication, traveler\'s diarrhea treatment, immunizations, and most topical medications',
                ],
            ],
            'rejection_message' => 'Based on your answers, you are not eligible for this Manulife insurance plan. Please contact us for alternative options.',
        ];
    }
    
    private function getManulifeStep2Config(): array
    {
        return [
            'title' => 'Rate Category C Triggers - Step 2',
            'description' => 'Questions that automatically place you in Rate Category C (highest premium)',
            'instructions' => 'Any "Yes" answer in this step means you will be placed in Rate Category C.',
            'questions' => [
                [
                    'id' => 'q2_1',
                    'code' => 'serious_conditions_ever',
                    'text' => 'Have you EVER been diagnosed with or treated for: a) Heart condition; b) Aortic aneurysm, cirrhosis of the liver, Parkinson\'s disease, or Alzheimer\'s/dementia?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q2_2',
                    'code' => 'high_bp_meds_3m',
                    'text' => 'In the last 3 months, have you been prescribed or taken 3 or more medications for high blood pressure?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q2_3',
                    'code' => 'recent_conditions_5y',
                    'text' => 'In the last 5 years, have you been diagnosed with, prescribed medication for, or treated for any of: Lung condition, Stroke/CVA or mini-stroke/TIA, Diabetes (treated with medication/insulin), Narrowed or blocked artery in the legs or neck?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
            ],
            'trigger_message' => 'You will be placed in Rate Category C (highest premium rate).',
        ];
    }
    
    private function getManulifeStep3Config(): array
    {
        return [
            'title' => 'Rate Category B Triggers - Step 3',
            'description' => 'Questions that place you in Rate Category B (medium premium)',
            'instructions' => 'Answer honestly. Any "Yes" answer places you in Rate Category B.',
            'questions' => [
                [
                    'id' => 'q3_1',
                    'code' => 'gastrointestinal_2y',
                    'text' => 'In the last 2 years, have you been diagnosed with, prescribed medication for, or treated for: Gastrointestinal bleeding, bowel obstruction, bowel surgery, chronic bowel disorder (Crohn\'s, ulcerative colitis)?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q3_2',
                    'code' => 'kidney_liver_pancreas_2y',
                    'text' => 'In the last 2 years, have you been diagnosed with, prescribed medication for, or treated for: Kidney disorder (including stones), liver disorder, or pancreatitis?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q3_3',
                    'code' => 'blood_disorder_2y',
                    'text' => 'In the last 2 years, have you been diagnosed with and/or treated by a hematologist or internist for a blood disorder?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q3_4',
                    'code' => 'recent_falls_6m',
                    'text' => 'Are you age 70 or older, and have you had a fall requiring medical attention in the last 6 months?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q3_5',
                    'code' => 'er_visits_6m',
                    'text' => 'In the last 6 months, have you received medical advice/treatment in an ER more than twice?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
            ],
            'trigger_message' => 'You will be placed in Rate Category B (medium premium rate).',
        ];
    }
    
    private function getManulifeStep4Config(): array
    {
        return [
            'title' => 'Smoking Information - Step 4',
            'description' => 'Smoking and vaping history for premium calculation',
            'instructions' => 'Honest answers required. Smoking/vaping adds 10% to your premium.',
            'questions' => [
                [
                    'id' => 'q4_1',
                    'code' => 'smoking_2y',
                    'text' => 'In the last 2 years, have you smoked cigarettes and/or used vaping products and/or e-cigarettes?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'q4_2',
                    'code' => 'years_smoking',
                    'text' => 'How many years have you been smoking? (If applicable)',
                    'type' => 'number',
                    'required' => false,
                    'min' => 0,
                    'max' => 100,
                    'conditional' => [
                        'depends_on' => 'smoking_2y',
                        'show_if' => 'yes',
                    ],
                ],
                [
                    'id' => 'q4_3',
                    'code' => 'uses_vaping',
                    'text' => 'Do you currently use vaping products or e-cigarettes?',
                    'type' => 'yes_no',
                    'required' => false,
                    'conditional' => [
                        'depends_on' => 'smoking_2y',
                        'show_if' => 'yes',
                    ],
                ],
            ],
            'surcharge_message' => 'Smoking/vaping adds 10% surcharge to your premium.',
        ];
    }
    
    private function getJfStep1Config(): array
    {
        return [
            'title' => 'Health Declaration - JF TOP',
            'description' => 'Basic health information for JF TOP Travel Insurance',
            'instructions' => 'JF TOP provides limited coverage for pre-existing conditions.',
            'questions' => [
                [
                    'id' => 'jf_q1_1',
                    'code' => 'pre_existing_conditions',
                    'text' => 'Do you have any pre-existing medical conditions that require regular treatment or medication?',
                    'type' => 'yes_no',
                    'required' => true,
                    'help_text' => 'JF TOP has limited coverage for pre-existing conditions. Contact us for details.',
                ],
                [
                    'id' => 'jf_q1_2',
                    'code' => 'recent_hospitalization_12m',
                    'text' => 'Have you been hospitalized or had surgery in the last 12 months?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'jf_q1_3',
                    'code' => 'planned_medical_treatment',
                    'text' => 'Are you traveling for planned medical treatment or surgery?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
            ],
            'rejection_message' => 'For complex medical conditions, please contact us for customized quotes.',
        ];
    }
    
    private function getJfStep4Config(): array
    {
        return [
            'title' => 'Lifestyle Information',
            'description' => 'Additional information for premium calculation',
            'instructions' => 'This helps us provide accurate coverage.',
            'questions' => [
                [
                    'id' => 'jf_q4_1',
                    'code' => 'smoking_status',
                    'text' => 'Do you smoke cigarettes?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'jf_q4_2',
                    'code' => 'general_health',
                    'text' => 'How would you describe your general health?',
                    'type' => 'select',
                    'required' => true,
                    'options' => [
                        ['value' => 'excellent', 'label' => 'Excellent - No health issues'],
                        ['value' => 'good', 'label' => 'Good - Minor issues, well controlled'],
                        ['value' => 'fair', 'label' => 'Fair - Some health concerns'],
                        ['value' => 'poor', 'label' => 'Poor - Significant health issues'],
                    ],
                ],
            ],
            'surcharge_message' => 'Smoking adds 15% surcharge to JF TOP premiums.',
        ];
    }
    
    private function getAllianzStep1Config(): array
    {
        return [
            'title' => 'Allianz Health Screening',
            'description' => 'Initial health screening for Allianz travel insurance',
            'instructions' => 'Answer all questions to determine eligibility.',
            'questions' => [
                [
                    'id' => 'alz_q1_1',
                    'code' => 'age_over_75',
                    'text' => 'Are you 75 years of age or older?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'alz_q1_2',
                    'code' => 'cancer_treatment_5y',
                    'text' => 'Have you had cancer treatment (chemotherapy, radiation, surgery) in the last 5 years?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
            ],
        ];
    }
    
    private function getAllianzStep2Config(): array
    {
        return [
            'title' => 'Medical Conditions - Allianz',
            'description' => 'Details of medical conditions',
            'instructions' => 'Provide information about any medical conditions.',
            'questions' => [
                [
                    'id' => 'alz_q2_1',
                    'code' => 'heart_condition',
                    'text' => 'Do you have any heart condition (angina, heart attack, bypass surgery, etc.)?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'alz_q2_2',
                    'code' => 'respiratory_condition',
                    'text' => 'Do you have any respiratory condition (COPD, asthma requiring hospitalization, etc.)?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
            ],
        ];
    }
    
    private function getAllianzStep4Config(): array
    {
        return [
            'title' => 'Additional Information',
            'description' => 'Final details for your application',
            'instructions' => 'Complete your application.',
            'questions' => [
                [
                    'id' => 'alz_q4_1',
                    'code' => 'tobacco_use',
                    'text' => 'Do you use tobacco products?',
                    'type' => 'yes_no',
                    'required' => true,
                ],
                [
                    'id' => 'alz_q4_2',
                    'code' => 'travel_assistance',
                    'text' => 'Do you require special travel assistance?',
                    'type' => 'yes_no',
                    'required' => true,
                    'help_text' => 'Wheelchair assistance, oxygen, etc.',
                ],
            ],
            'surcharge_message' => 'Tobacco use adds 12% surcharge to Allianz premiums.',
        ];
    }
}