<?php

namespace App\Services;

use App\Enums\QuestionnaireStep;
use App\Enums\RateCategory;

class QuestionnaireService
{
    public function getQuestionsForStep(QuestionnaireStep $step): array
    {
        return match ($step) {
            QuestionnaireStep::ELIGIBILITY => $this->getEligibilityQuestions(),
            QuestionnaireStep::RATE_C_TRIGGER => $this->getRateCQuestions(),
            QuestionnaireStep::RATE_B_TRIGGER => $this->getRateBQuestions(),
            QuestionnaireStep::SMOKING => $this->getSmokingQuestions(),
        };
    }

    public function evaluateAnswers(array $answers): array
    {
        $result = [
            'is_eligible' => true,
            'rate_category' => RateCategory::A,
            'has_smoking_surcharge' => false,
        ];

        // Step 1: Eligibility check
        if ($this->hasIneligibleAnswer($answers['step1'] ?? [])) {
            $result['is_eligible'] = false;
            return $result;
        }

        // Step 2: Rate C triggers
        if ($this->hasRateCTrigger($answers['step2'] ?? [])) {
            $result['rate_category'] = RateCategory::C;
            return $result;
        }

        // Step 3: Rate B triggers
        if ($this->hasRateBTrigger($answers['step3'] ?? [])) {
            $result['rate_category'] = RateCategory::B;
        }

        // Step 4: Smoking
        if ($this->hasSmoking($answers['step4'] ?? [])) {
            $result['has_smoking_surcharge'] = true;
        }

        return $result;
    }

    private function getEligibilityQuestions(): array
    {
        return [
            [
                'code' => 'heart_surgery_10y',
                'text' => 'Have you had a heart bypass, coronary angioplasty, or heart valve surgery more than 10 years ago?',
                'type' => 'yes_no',
            ],
            [
                'code' => 'multiple_conditions',
                'text' => 'In the last 3 years, have you been diagnosed with, prescribed medication for, or treated for ANY TWO of the following conditions?',
                'type' => 'yes_no',
            ],
        ];
    }

    private function getRateCQuestions(): array
    {
        return [
            [
                'code' => 'heart_condition',
                'text' => 'Have you ever been diagnosed with or treated for a heart condition?',
                'type' => 'yes_no',
            ],
            [
                'code' => 'serious_conditions',
                'text' => 'Have you ever been diagnosed with any of: aortic aneurysm, cirrhosis of the liver, Parkinson\'s disease, Alzheimer\'s/dementia?',
                'type' => 'yes_no',
            ],
        ];
    }

    private function getRateBQuestions(): array
    {
        return [
            [
                'code' => 'gastrointestinal',
                'text' => 'In the last 2 years, have you been diagnosed with, prescribed medication for, or treated for: Gastrointestinal bleeding, bowel obstruction, or bowel surgery?',
                'type' => 'yes_no',
            ],
            [
                'code' => 'er_visits',
                'text' => 'In the last 6 months, have you received medical advice/treatment in an ER more than twice?',
                'type' => 'yes_no',
            ],
        ];
    }

    private function getSmokingQuestions(): array
    {
        return [
            [
                'code' => 'smoking_2y',
                'text' => 'In the last 2 years, have you smoked cigarettes and/or used vaping products and/or e-cigarettes?',
                'type' => 'yes_no',
            ],
        ];
    }

    private function hasIneligibleAnswer(array $answers): bool
    {
        foreach ($answers as $answer) {
            if ($answer === true) {
                return true;
            }
        }
        return false;
    }

    private function hasRateCTrigger(array $answers): bool
    {
        foreach ($answers as $answer) {
            if ($answer === true) {
                return true;
            }
        }
        return false;
    }

    private function hasRateBTrigger(array $answers): bool
    {
        foreach ($answers as $answer) {
            if ($answer === true) {
                return true;
            }
        }
        return false;
    }

    private function hasSmoking(array $answers): bool
    {
        foreach ($answers as $answer) {
            if ($answer === true) {
                return true;
            }
        }
        return false;
    }
}