<?php

namespace App\Http\Controllers\Inquiry;

use App\Http\Controllers\ApiController;
use App\Http\Requests\Inquiry\GetFormOptionsRequest;
use App\Http\Resources\Inquiry\CityTranslationResource;
use App\Http\Resources\Inquiry\CountryTranslationResource;
use App\Http\Resources\Inquiry\ProvinceTranslationResource;
use App\Models\CityTranslation;
use App\Models\CountryTranslation;
use App\Models\Language;
use App\Models\Price;
use App\Models\ProvinceTranslation;
use Illuminate\Http\JsonResponse;
use Throwable;

/**
 * @OA\Options(
 *   path="/api/inquiry",
 *   tags={"Inquiry"},
 *   summary="Get Inquiry Form Options",
 *   operationId="inquiry_options",
 *   security={{"auth":{}}},
 *
 *   @OA\Parameter(name="Accept-Language", in="header", description="Accept-Language", required=true, example="en", @OA\Schema(type="string")),
 *   @OA\Parameter(name="API-Key", in="header", description="API-Key", required=true, example="", @OA\Schema(type="string")),
 *
 *   @OA\Response(response=200, description="success")
 * )
 */
class GetFormOptionsController extends ApiController
{
    public function __invoke(GetFormOptionsRequest $request): JsonResponse
    {
        try {
            $data = $this->sanitize($request, [
                'language' => $request->validated('language'),
                'token'    => $request->validated('token'),
            ]);
            $languageId = Language::where('code', $data['language'])->value('id');
            $insuranceCovers = Price::query()
                ->select('insurance_cover')
                ->distinct()
                ->where('is_active', 1)
                ->orderBy('insurance_cover')
                ->pluck('insurance_cover')
                ->toArray();
            $countries = CountryTranslationResource::collection(CountryTranslation::query()->where(['language_id' => $languageId, 'is_active' => 1])->get());
            $provinces = ProvinceTranslationResource::collection(ProvinceTranslation::query()->where(['language_id' => $languageId, 'is_active' => 1])->get());
            $cities = CityTranslationResource::collection(CityTranslation::query()->where(['language_id' => $languageId, 'is_active' => 1])->get());
            return $this->success([
                'insurance_covers' => $insuranceCovers,
                'countries'        => $countries,
                'provinces'        => $provinces,
                'cities'           => $cities,
            ]);
        } catch (Throwable $e) {
            return $this->error($e->getMessage(), status: $e->getCode());
        }
    }
}
