<?php

namespace App\Filament\SuperAdmin\Resources;

use App\Filament\Custom\Actions;
use App\Filament\Custom\Columns;
use App\Filament\SuperAdmin\Resources\AdminResource\Pages;
use App\Models\Admin;
use App\Providers\Filament\SuperAdminPanelProvider;
use Exception;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Filament\Actions\EditAction;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use UnitEnum;

class AdminResource extends Resource
{
    protected static ?string $model = Admin::class;

    protected static string|UnitEnum|null $navigationGroup = SuperAdminPanelProvider::USERS;

    protected static ?int $navigationSort = 1;

    protected static ?string $recordTitleAttribute = 'username';

    /**
     * @throws Exception
     */
    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('username')
                    ->unique(ignoreRecord: true)
                    ->required(),
                TextInput::make('email')
                    ->email()
                    ->unique(ignoreRecord: true)
                    ->nullable(),
                TextInput::make('password')
                    ->password()
                    ->dehydrateStateUsing(fn($state) => Hash::make($state))
                    ->dehydrated(fn($state) => filled($state))
                    ->required(fn(string $context): bool => $context === 'create'),
                TextInput::make('name')
                    ->required(),
                TextInput::make('domain')
                    ->label('Domain')
                    ->required()
                    ->unique(ignoreRecord: true)
                    ->rule('url')
                    ->validationMessages([
                        'unique' => 'This domain is already registered.',
                    ]),
                TextInput::make('license')
                    ->label('License')
                    ->helperText('If empty, it will be generated automatically.')
                    ->nullable()
                    ->default(fn($record) => $record->license ?? Str::upper(Str::random(12))),
                Checkbox::make('is_active')
                    ->label('Active')
                    ->inline(false)
                    ->default(true),
            ]);
    }

    /**
     * @throws Exception
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('row')
                    ->label('#')
                    ->rowIndex(),
                TextColumn::make('username')
                    ->searchable(isIndividual: true, isGlobal: false)
                    ->sortable(),
                TextColumn::make('name')
                    ->searchable(isIndividual: true, isGlobal: false)
                    ->sortable(),
                TextColumn::make('email')
                    ->searchable(isIndividual: true, isGlobal: false)
                    ->sortable(),
                TextColumn::make('domain')
                    ->label('Domain')
                    ->toggleable(),
                TextColumn::make('license')
                    ->label('License')
                    ->toggleable(),
                Columns::toggleIcon('is_active'),
            ])
            ->filters([])
            ->recordActions([
                EditAction::make(),
            ])
            ->toolbarActions([]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListAdmins::route('/'),
            'create' => Pages\CreateAdmin::route('/create'),
            'edit'   => Pages\EditAdmin::route('/{record}/edit'),
        ];
    }
}
