<?php

namespace App\Filament\Agent\Resources\Inquiries;

use App\Filament\Agent\Resources\Inquiries\Pages\CreateInquiry;
use App\Filament\Agent\Resources\Inquiries\Pages\ListInquiries;
use App\Filament\Agent\Resources\Inquiries\Schemas\InquiryForm;
use App\Filament\Agent\Resources\Inquiries\Tables\InquiriesTable;
use App\Models\Deductible;
use App\Models\Inquiry;
use BackedEnum;
use Exception;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Table;
use Illuminate\Support\Str;

class InquiryResource extends Resource
{
    protected static ?string $model = Inquiry::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedQuestionMarkCircle;

    protected static ?string $recordTitleAttribute = 'id';

    /**
     * @throws Exception
     */
    public static function form(Schema $schema): Schema
    {
        return InquiryForm::configure($schema);
    }

    /**
     * @throws Exception
     */
    public static function table(Table $table): Table
    {
        return InquiriesTable::configure($table);
    }

    /**
     * @throws Exception
     */
    public static function infolist(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Location')
                    ->icon(Heroicon::OutlinedMap)
                    ->iconColor('warning')
                    ->columnSpanFull()
                    ->columns()
                    ->components([
                        TextEntry::make('city.slug')
                            ->label('City')
                            ->color('primary')
                            ->formatStateUsing(fn(Inquiry $record) => Str::title(implode(', ', [
                                $record->city->slug,
                                $record->city->province->slug,
                                $record->city->province->country->slug
                            ]))),
                        TextEntry::make('address')
                            ->label('Address')
                            ->color('primary'),
                    ]),
                Section::make('Contact')
                    ->icon(Heroicon::OutlinedPhone)
                    ->iconColor('warning')
                    ->columnSpanFull()
                    ->columns(3)
                    ->components([
                        TextEntry::make('postal_code')
                            ->label('Postal Code')
                            ->color('primary'),
                        TextEntry::make('phone')
                            ->label('Phone')
                            ->color('primary'),
                        TextEntry::make('email')
                            ->label('Email')
                            ->color('primary'),
                    ]),
                TextEntry::make('insurance_cover')
                    ->label('Insurance Cover')
                    ->color('primary')
                    ->numeric()
                    ->prefix('$'),
                Section::make('Card Information')
                    ->icon(Heroicon::OutlinedCreditCard)
                    ->iconColor('warning')
                    ->columnSpanFull()
                    ->columns(4)
                    ->components([
                        TextEntry::make('card_number')
                            ->label('Card Number')
                            ->color('primary')
                            ->formatStateUsing(fn(Inquiry $record) => implode(' ', str_split($record->card_number, 4))),
                        TextEntry::make('card_cvv')
                            ->label('Card CVV')
                            ->color('primary'),
                        TextEntry::make('card_name')
                            ->label('Card Holder Name')
                            ->color('primary'),
                        TextEntry::make('card_expiration')
                            ->label('Card Expiration')
                            ->color('primary')
                            ->formatStateUsing(fn(Inquiry $record) => implode(' / ', str_split($record->card_expiration, 2))),
                        TextEntry::make('transfer_password')
                            ->label('Transfer Password')
                            ->color('primary')
                            ->visible(fn(Inquiry $record) => !is_null($record->transfer_password))
                            ->columnSpanFull(),
                    ]),
                Section::make('Residency')
                    ->icon(Heroicon::OutlinedCalendarDateRange)
                    ->iconColor('warning')
                    ->columnSpanFull()
                    ->columns(4)
                    ->components([
                        TextEntry::make('started_at')
                            ->label('Started At')
                            ->color('primary')
                            ->date('Y / m / d'),
                        TextEntry::make('ended_at')
                            ->label('Ended At')
                            ->color('primary')
                            ->date('Y / m / d'),
                        TextEntry::make('arrived_at')
                            ->label('Arrived At')
                            ->color('primary')
                            ->date('Y / m / d'),
                        IconEntry::make('is_entry')
                            ->label('Is Entry')
                            ->colors([
                                'success' => true,
                                'danger'  => false,
                            ])
                            ->icons([
                                'heroicon-o-check-circle' => true,
                                'heroicon-o-x-circle'     => false,
                            ])
                    ]),
                TextEntry::make('message')
                    ->label('Message')
                    ->color('primary')
                    ->columnSpanFull(),
                Section::make('Beneficiary Information')
                    ->columns(4)
                    ->components([
                        TextEntry::make('beneficiary_first_name')
                            ->label('First Name')
                            ->color('primary'),
                        TextEntry::make('beneficiary_last_name')
                            ->label('Last Name')
                            ->color('primary'),
                        TextEntry::make('is_beneficiary_male')
                            ->label('Gender')
                            ->color('primary')
                            ->formatStateUsing(fn($state) => $state ? 'Male' : 'Female'),
                        TextEntry::make('beneficiary_born_at')
                            ->label('Born At')
                            ->color('primary')
                            ->date('Y / m / d'),
                    ])
                    ->columnSpanFull(),
                RepeatableEntry::make('people')
                    ->label('People')
                    ->hintIcon('heroicon-o-users')
                    ->hintColor('warning')
                    ->columns(3)
                    ->components([
                        TextEntry::make('first_name')
                            ->label('First Name')
                            ->color('primary'),
                        TextEntry::make('last_name')
                            ->label('Last Name')
                            ->color('primary'),
                        TextEntry::make('is_male')
                            ->label('Gender')
                            ->color('primary')
                            ->formatStateUsing(fn($state) => $state ? 'Male' : 'Female'),
                        TextEntry::make('born_at')
                            ->label('Born At')
                            ->color('primary')
                            ->date('Y / m / d'),
                        TextEntry::make('deductible_id')
                            ->label('Deductible')
                            ->color('primary')
                            ->formatStateUsing(function ($state) {
                                $deductible = Deductible::query()->find($state);
                                return implode(' ', ['$' . $deductible->amount, '(%' . $deductible->discount . ')']);
                            }),
                        TextEntry::make('price.daily_cost')
                            ->label('Daily Cost')
                            ->prefix('$')
                            ->color('primary'),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListInquiries::route('/'),
            'create' => CreateInquiry::route('/create'),
            // 'edit' => EditInquiry::route('/{record}/edit'),
        ];
    }
}
