// app/Filament/Admin/Resources/InquiryResource/InquiryResource.php
<?php

namespace App\Filament\Admin\Resources;

use App\Filament\Admin\Resources\InquiryResource\Pages;
use App\Models\Inquiry;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class InquiryResource extends Resource
{
    protected static ?string $model = Inquiry::class;
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $navigationGroup = 'Insurance Management';
    protected static ?int $navigationSort = 1;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Inquiry Information')
                    ->schema([
                        Forms\Components\TextInput::make('reference_number')
                            ->label('Reference Number')
                            ->disabled()
                            ->dehydrated(false),
                            
                        Forms\Components\Select::make('status')
                            ->options([
                                'pending' => 'Pending',
                                'under_review' => 'Under Review',
                                'approved' => 'Approved',
                                'rejected' => 'Rejected',
                                'completed' => 'Completed'
                            ])
                            ->required(),
                            
                        Forms\Components\Select::make('agent_id')
                            ->relationship('agent', 'name')
                            ->searchable()
                            ->required(),
                            
                        Forms\Components\Toggle::make('is_entry')
                            ->label('Entry Insurance')
                            ->onColor('success')
                            ->offColor('danger'),
                            
                        Forms\Components\Toggle::make('is_deal')
                            ->label('Is Deal')
                            ->onColor('success'),
                    ])->columns(2),
                    
                Forms\Components\Section::make('Coverage Details')
                    ->schema([
                        Forms\Components\TextInput::make('insurance_cover')
                            ->numeric()
                            ->suffix('$')
                            ->required(),
                            
                        Forms\Components\TextInput::make('total_premium')
                            ->numeric()
                            ->prefix('$')
                            ->required(),
                            
                        Forms\Components\TextInput::make('number_of_people')
                            ->numeric()
                            ->required(),
                            
                        Forms\Components\DatePicker::make('started_at')
                            ->required(),
                            
                        Forms\Components\DatePicker::make('ended_at')
                            ->required(),
                    ])->columns(3),
                    
                Forms\Components\Section::make('Contact Information')
                    ->schema([
                        Forms\Components\TextInput::make('email')
                            ->email()
                            ->required(),
                            
                        Forms\Components\TextInput::make('phone')
                            ->tel()
                            ->required(),
                            
                        Forms\Components\Textarea::make('address')
                            ->rows(2)
                            ->required(),
                            
                        Forms\Components\TextInput::make('postal_code')
                            ->required(),
                    ])->columns(2),
                    
                Forms\Components\Section::make('Payment Information')
                    ->schema([
                        Forms\Components\TextInput::make('card_name')
                            ->label('Card Holder Name'),
                            
                        Forms\Components\TextInput::make('card_number')
                            ->label('Card Number')
                            ->mask('9999-9999-9999-9999'),
                            
                        Forms\Components\TextInput::make('card_expiration')
                            ->label('Expiration (MM/YY)'),
                            
                        Forms\Components\Textarea::make('message')
                            ->rows(3),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('reference_number')
                    ->label('Ref No.')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('agent.name')
                    ->label('Agent')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'pending' => 'warning',
                        'under_review' => 'info',
                        'approved' => 'success',
                        'rejected' => 'danger',
                        'completed' => 'primary',
                        default => 'gray',
                    })
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('is_entry')
                    ->label('Type')
                    ->boolean()
                    ->trueIcon('heroicon-o-arrow-right')
                    ->falseIcon('heroicon-o-arrow-left')
                    ->trueColor('success')
                    ->falseColor('warning')
                    ->trueTooltip('Entry Insurance')
                    ->falseTooltip('Exit Insurance'),
                    
                Tables\Columns\TextColumn::make('number_of_people')
                    ->label('People')
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('total_premium')
                    ->label('Premium')
                    ->money('USD')
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('inquired_at')
                    ->label('Submitted')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                    
                Tables\Columns\TextColumn::make('local_submitted_at')
                    ->label('Local Time')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'pending' => 'Pending',
                        'under_review' => 'Under Review',
                        'approved' => 'Approved',
                        'rejected' => 'Rejected',
                        'completed' => 'Completed',
                    ]),
                    
                Tables\Filters\SelectFilter::make('is_entry')
                    ->label('Insurance Type')
                    ->options([
                        '1' => 'Entry to Canada',
                        '0' => 'Exit from Canada',
                    ]),
                    
                Tables\Filters\Filter::make('inquired_at')
                    ->form([
                        Forms\Components\DatePicker::make('from'),
                        Forms\Components\DatePicker::make('until'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['from'], fn($q) => $q->whereDate('inquired_at', '>=', $data['from']))
                            ->when($data['until'], fn($q) => $q->whereDate('inquired_at', '<=', $data['until']));
                    }),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('inquired_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            // People relation will be added later
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListInquiries::route('/'),
            'create' => Pages\CreateInquiry::route('/create'),
            'view' => Pages\ViewInquiry::route('/{record}'),
            'edit' => Pages\EditInquiry::route('/{record}/edit'),
        ];
    }
    public static function getRelations(): array
{
    return [
        PeopleRelationManager::class,
    ];
}
}