# Developer Onboarding Guide

Welcome to the Insurance Application project! This guide will help you get up and running quickly.

## 📋 Table of Contents

1. [Project Overview](#project-overview)
2. [Prerequisites](#prerequisites)
3. [Initial Setup](#initial-setup)
4. [Development Workflow](#development-workflow)
5. [Project Structure](#project-structure)
6. [Key Technologies](#key-technologies)
7. [Common Tasks](#common-tasks)
8. [Testing](#testing)
9. [API Documentation](#api-documentation)
10. [Troubleshooting](#troubleshooting)

## 🎯 Project Overview

This is an insurance inquiry/quote management system built with Laravel 12. The application allows users to:
- Submit insurance inquiries
- View supported insurance plans
- Manage insurance companies, plans, and pricing
- Handle multi-language content (with Persian/Jalali date support)
- Admin panel for managing the system

### Core Features
- **Authentication**: JWT-based API authentication
- **Inquiry System**: Submit and manage insurance inquiries
- **Multi-language**: Support for multiple languages with translations
- **Admin Panel**: Filament-based admin interface
- **Geographic Data**: Countries, provinces, and cities management
- **Pricing System**: Dynamic pricing for insurance plans

## 🔧 Prerequisites

Before you begin, ensure you have the following installed:

- **PHP**: 8.2 or higher
- **Composer**: Latest version
- **Node.js**: 18.x or higher (for frontend assets)
- **npm** or **yarn**: Package manager
- **Database**: MySQL/PostgreSQL/MariaDB
- **Git**: Version control

### Optional but Recommended
- **Laravel Sail**: For Docker-based development
- **Laravel Pail**: For log viewing
- **IDE**: PHPStorm, VS Code, or similar

## 🚀 Initial Setup

### 1. Clone the Repository

```bash
git clone <repository-url>
cd insurance
```

### 2. Install PHP Dependencies

```bash
composer install
```

### 3. Install Node Dependencies

```bash
npm install
```

### 4. Environment Configuration

Copy the environment file and configure it:

```bash
cp .env.example .env
php artisan key:generate
```

Edit `.env` with your local settings:
- Database credentials
- JWT secret (if needed)
- App URL
- Queue connection (if using queues)

### 5. Database Setup

```bash
# Run migrations
php artisan migrate

# (Optional) Seed the database
php artisan db:seed
```

### 6. Build Frontend Assets

For development:
```bash
npm run dev
```

For production:
```bash
npm run build
```

### 7. Start Development Server

**Option A: Using Composer Script (Recommended)**
```bash
composer run dev
```
This starts:
- Laravel server (port 8000)
- Queue worker
- Log viewer (Pail)
- Vite dev server

**Option B: Manual Start**
```bash
# Terminal 1: Laravel server
php artisan serve

# Terminal 2: Queue worker
php artisan queue:listen

# Terminal 3: Frontend assets
npm run dev
```

### 8. Access the Application

- **API**: http://localhost:8000/api
- **Admin Panel**: http://localhost:8000/admin (Filament)
- **API Documentation**: http://localhost:8000/api/documentation (Swagger)

## 📁 Project Structure

```
insurance/
├── app/
│   ├── Filament/          # Filament admin panel resources
│   ├── Helpers/           # Helper classes
│   ├── Http/
│   │   ├── Controllers/   # API controllers
│   │   │   ├── Authentication/
│   │   │   ├── Base/
│   │   │   └── Inquiry/
│   │   └── Middleware/    # Custom middleware
│   ├── Models/            # Eloquent models
│   ├── Policies/          # Authorization policies
│   └── Providers/         # Service providers
├── config/                # Configuration files
├── database/
│   ├── migrations/        # Database migrations
│   └── seeders/           # Database seeders
├── public/                # Public assets
├── resources/
│   ├── css/               # Stylesheets
│   └── js/                # JavaScript files
├── routes/
│   ├── api.php            # API routes
│   └── web.php            # Web routes
├── storage/               # Storage (logs, cache, etc.)
└── tests/                 # Test files
```

## 🛠 Key Technologies

### Backend
- **Laravel 12**: PHP framework
- **Filament 4.0**: Admin panel
- **JWT (lcobucci/jwt)**: Authentication
- **Swagger (l5-swagger)**: API documentation
- **Jalali Dates (morilog/jalali)**: Persian calendar support

### Frontend
- **Vite**: Build tool
- **Tailwind CSS 4.0**: Styling
- **Axios**: HTTP client

### Development Tools
- **Laravel Pint**: Code formatting
- **PHPUnit**: Testing
- **Laravel Pail**: Log viewer
- **Laravel Sail**: Docker environment

## 📝 Common Tasks

### Running Tests

```bash
composer run test
# or
php artisan test
```

### Code Formatting

```bash
./vendor/bin/pint
```

### Creating a New Migration

```bash
php artisan make:migration create_example_table
```

### Creating a New Model

```bash
php artisan make:model Example
```

### Creating a New Controller

```bash
php artisan make:controller ExampleController
```

### Creating a Filament Resource

```bash
php artisan make:filament-resource Example
```

### Clearing Cache

```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### Queue Jobs

```bash
# Process queue jobs
php artisan queue:work

# Listen for jobs (development)
php artisan queue:listen
```

## 🧪 Testing

### Running Tests

```bash
# Run all tests
php artisan test

# Run specific test file
php artisan test --filter ExampleTest

# Run with coverage
php artisan test --coverage
```

### Writing Tests

Tests are located in `tests/` directory. Follow Laravel testing conventions:
- Feature tests: `tests/Feature/`
- Unit tests: `tests/Unit/`

Example:
```php
<?php

namespace Tests\Feature;

use Tests\TestCase;

class ExampleTest extends TestCase
{
    public function test_example(): void
    {
        $response = $this->get('/api/base/health');
        $response->assertStatus(200);
    }
}
```

## 📚 API Documentation

### Authentication Endpoints

- `GET /api/auth/check` - Check authentication status
- `POST /api/auth/login` - Login and get JWT token
- `PATCH /api/auth/refresh` - Refresh JWT token

### Base Endpoints

- `GET /api/base/health` - Health check
- `GET /api/base/content` - Get content (requires auth)

### Inquiry Endpoints

- `GET /api/inquiry/` - Get supported plans (requires auth)
- `POST /api/inquiry/` - Submit inquiry (requires auth)
- `OPTIONS /api/inquiry/` - Get form options (requires auth)

### API Documentation

Swagger documentation is available at:
- **URL**: `/api/documentation`
- **Generate docs**: `php artisan l5-swagger:generate`

## 🔍 Key Models

### Core Models
- **Admin**: Admin users
- **Agent**: Insurance agents
- **Company**: Insurance companies
- **Plan**: Insurance plans
- **Inquiry**: Customer inquiries
- **Person**: Person information
- **Price**: Pricing information
- **Deductible**: Deductible options

### Geographic Models
- **Country**: Countries
- **Province**: Provinces/States
- **City**: Cities

### Translation Models
- **CompanyTranslation**: Company translations
- **PlanTranslation**: Plan translations
- **DeductibleTranslation**: Deductible translations
- **CountryTranslation**: Country translations
- **ProvinceTranslation**: Province translations
- **CityTranslation**: City translations

## 🐛 Troubleshooting

### Common Issues

**Issue: Composer install fails**
```bash
# Clear composer cache
composer clear-cache
composer install --no-cache
```

**Issue: Permission errors**
```bash
# Fix storage permissions
chmod -R 775 storage bootstrap/cache
```

**Issue: Database connection error**
- Check `.env` database credentials
- Ensure database exists
- Verify database server is running

**Issue: JWT token errors**
- Regenerate JWT secret: `php artisan jwt:secret`
- Clear config cache: `php artisan config:clear`

**Issue: Frontend assets not loading**
```bash
# Rebuild assets
npm run build
# or for development
npm run dev
```

**Issue: Queue jobs not processing**
- Ensure queue worker is running: `php artisan queue:work`
- Check queue connection in `.env`
- Verify database connection

## 📖 Additional Resources

### Laravel Documentation
- [Laravel 12 Docs](https://laravel.com/docs/12.x)
- [Laravel API Documentation](https://laravel.com/api/12.x)

### Filament Documentation
- [Filament 4.0 Docs](https://filamentphp.com/docs/4.x)

### Project-Specific
- Check `routes/api.php` for API endpoints
- Review `app/Models/` for data structure
- See `database/migrations/` for database schema

## 🤝 Getting Help

1. **Check Documentation**: Review this guide and Laravel docs
2. **Code Review**: Look at existing code for patterns
3. **Team Communication**: Reach out to team lead or senior developers
4. **Logs**: Check `storage/logs/laravel.log` for errors

## ✅ Onboarding Checklist

- [ ] Repository cloned
- [ ] Dependencies installed (Composer & npm)
- [ ] Environment file configured
- [ ] Database migrated and seeded
- [ ] Development server running
- [ ] Admin panel accessible
- [ ] API documentation reviewed
- [ ] Tests passing
- [ ] Code formatting tool (Pint) configured
- [ ] IDE configured with Laravel support

---

**Welcome to the team! 🎉**

For questions or issues, contact your team lead.

